#include <math.h>
#include <gamelib/map.h>

struct MapData* map[MAP_X_SIZE][MAP_Y_SIZE];

void initialize_map(){
  int map_x, map_y;
  
  for( map_x = 0; map_x < MAP_X_SIZE; ++map_x ){
    for( map_y = 0; map_y < MAP_Y_SIZE; ++map_y ){
      map[map_x][map_y] = NULL;
    }
  }
}

void destroy_map(){
  int map_x, map_y, wall_index;
  for( map_x = 0; map_x < MAP_X_SIZE; ++map_x ){
    for( map_y = 0; map_y < MAP_Y_SIZE; ++map_y ){
      if( NULL != map[map_x][map_y] ){
	for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
	  dGeomSetData( map[map_x][map_y]->wall_data[wall_index].geom_ID, NULL );
	  dBodyDestroy( map[map_x][map_y]->wall_data[wall_index].body_ID );
	  dGeomDestroy( map[map_x][map_y]->wall_data[wall_index].geom_ID );
	  dJointDestroy( map[map_x][map_y]->wall_data[wall_index].fixed_joint_ID );
	}
	free( map[map_x][map_y] );
	map[map_x][map_y] = NULL;
      }
    }
  }
}

int make_strait_walls( int map_x, int map_y, enum DIRECTION direction ){
  static const int wall_size = 2;
  dReal wall_position[wall_size][2];
  dReal wall_direction[wall_size][2];
  int wall_index;
  if( map[map_x][map_y] == NULL ){
    map[map_x][map_y] = (struct MapData*)malloc( sizeof(struct MapData) );
    for( wall_index = 0; wall_index < MAP_ONE_SQUIRE_MAX_WALL_SIZE; ++wall_index ){
      map[map_x][map_y]->wall_object[wall_index].object_type = WALL;
      map[map_x][map_y]->wall_object[wall_index].data = &map[map_x][map_y]->wall_data[wall_index];
    }
  }else{
    return -1; // if the map on the position is filled, then return -1 (error).
  }
  
  map[map_x][map_y]->center_position[0] = ( (dReal)map_x + 0.5 ) * MAP_SQUIRE_WIDTH;
  map[map_x][map_y]->center_position[1] = ( (dReal)map_y + 0.5 ) * MAP_SQUIRE_WIDTH;

  map[map_x][map_y]->wall_size = wall_size;

  wall_position[0][0] = ROAD_WIDTH/2;
  wall_position[0][1] = 0.0;
  wall_direction[0][0] = 0.0;
  wall_direction[0][1] = 1.0;

  wall_position[1][0] = -ROAD_WIDTH/2;
  wall_position[1][1] = 0.0;
  wall_direction[1][0] = 0.0;
  wall_direction[1][1] = 1.0;

  switch( direction ){
  case FROM_EAST_TO_WEST:
  case FROM_WEST_TO_EAST:
    { /* to rotate 90 deg crockwise*/
      for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
	rotate_90deg_clockwise( wall_position[wall_index]);
	rotate_90deg_clockwise( wall_direction[wall_index]);
      }
    }
  case FROM_SOUTH_TO_NORTH:
  case FROM_NORTH_TO_SOUTH:
  default:
    break;
  }


  for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
    make_wall( map[map_x][map_y]->center_position[0] + wall_position[wall_index][0], /* wall center position x */
	       map[map_x][map_y]->center_position[1] + wall_position[wall_index][1], /* wall center position y*/
	       wall_direction[wall_index][0], /* wall direction x */
	       wall_direction[wall_index][1], /* wall direction y */
	       MAP_SQUIRE_WIDTH,
	       &(map[map_x][map_y]->wall_object[wall_index]) );
  }

  return 0;
}

int make_turn_walls( int map_x, int map_y, enum DIRECTION direction ){
  static const int wall_size = 4;
  dReal wall_position[wall_size][2];
  dReal wall_direction[wall_size][2];
  dReal wall_length[wall_size];
  int wall_index;
  if( map[map_x][map_y] == NULL ){
    map[map_x][map_y] = (struct MapData*)malloc( sizeof(struct MapData) );
    for( wall_index = 0; wall_index < MAP_ONE_SQUIRE_MAX_WALL_SIZE; ++wall_index ){
      map[map_x][map_y]->wall_object[wall_index].object_type = WALL;
      map[map_x][map_y]->wall_object[wall_index].data = &map[map_x][map_y]->wall_data[wall_index];
    }
  }else{
    return -1; // if the map on the position is filled, then return -1 (error).
  }
  
  map[map_x][map_y]->center_position[0] = ( (dReal)map_x + 0.5 ) * MAP_SQUIRE_WIDTH;
  map[map_x][map_y]->center_position[1] = ( (dReal)map_y + 0.5 ) * MAP_SQUIRE_WIDTH;

  map[map_x][map_y]->wall_size = wall_size;

  /* the original is right turn (from south to east)*/
  wall_position[0][0] = -ROAD_WIDTH/2;
  wall_position[0][1] = -(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_direction[0][0] = 0.0;
  wall_direction[0][1] = 1.0;
  wall_length[0] = ROAD_WIDTH+(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;

  wall_position[1][0] = +(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_position[1][1] = +ROAD_WIDTH/2;
  wall_direction[1][0] = 1.0;
  wall_direction[1][1] = 0.0;
  wall_length[1] = ROAD_WIDTH+(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;
  
  
  wall_position[2][0] = +ROAD_WIDTH/2;
  wall_position[2][1] = -ROAD_WIDTH/2-(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_direction[2][0] = 0.0;
  wall_direction[2][1] = 1.0;
  wall_length[2] = (MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;

  wall_position[3][0] = +ROAD_WIDTH/2+(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_position[3][1] = -ROAD_WIDTH/2;
  wall_direction[3][0] = 1.0;
  wall_direction[3][1] = 0.0;
  wall_length[3] = (MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;


  switch( direction ){
  case FROM_EAST_TO_NORTH:
  case FROM_NORTH_TO_EAST:
    for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
      rotate_90deg_anticlockwise( wall_position[wall_index]);
      rotate_90deg_anticlockwise( wall_direction[wall_index]);
    }
    break;
  case FROM_NORTH_TO_WEST:
  case FROM_WEST_TO_NORTH:
    for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
      rotate_180deg( wall_position[wall_index]);
      rotate_180deg( wall_direction[wall_index]);
    }
    break;
  case FROM_WEST_TO_SOUTH:
  case FROM_SOUTH_TO_WEST:
    /* to rotate 90 deg crockwise*/
    for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
      rotate_90deg_clockwise( wall_position[wall_index]);
      rotate_90deg_clockwise( wall_direction[wall_index]);
    }
    break;
  case FROM_SOUTH_TO_EAST:
  case FROM_EAST_TO_SOUTH:
  default:
    break;
  }


  for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
    make_wall( map[map_x][map_y]->center_position[0] + wall_position[wall_index][0], /* wall center position x */
	       map[map_x][map_y]->center_position[1] + wall_position[wall_index][1], /* wall center position y*/
	       wall_direction[wall_index][0], /* wall direction x */
	       wall_direction[wall_index][1], /* wall direction y */
	       wall_length[wall_index],
	       &(map[map_x][map_y]->wall_object[wall_index]) );
  }

  return 0;
}

int make_t_junction_walls( int map_x, int map_y, enum DIRECTION direction ){
  static const int wall_size = 5;
  dReal wall_position[wall_size][2];
  dReal wall_direction[wall_size][2];
  dReal wall_length[wall_size];
  int wall_index;
  if( map[map_x][map_y] == NULL ){
    map[map_x][map_y] = (struct MapData*)malloc( sizeof(struct MapData) );
    for( wall_index = 0; wall_index < MAP_ONE_SQUIRE_MAX_WALL_SIZE; ++wall_index ){
      map[map_x][map_y]->wall_object[wall_index].object_type = WALL;
      map[map_x][map_y]->wall_object[wall_index].data = &map[map_x][map_y]->wall_data[wall_index];
    }
  }else{
    return -1; // if the map on the position is filled, then return -1 (error).
  }
  map[map_x][map_y]->center_position[0] = ( (dReal)map_x + 0.5 ) * MAP_SQUIRE_WIDTH;
  map[map_x][map_y]->center_position[1] = ( (dReal)map_y + 0.5 ) * MAP_SQUIRE_WIDTH;

  map[map_x][map_y]->wall_size = wall_size;

  wall_position[0][0] = +ROAD_WIDTH/2;
  wall_position[0][1] = -ROAD_WIDTH/2-(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_direction[0][0] = 0.0;
  wall_direction[0][1] = 1.0;
  wall_length[0] = (MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;

  wall_position[1][0] = +ROAD_WIDTH/2+(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_position[1][1] = -ROAD_WIDTH/2;
  wall_direction[1][0] = 1.0;
  wall_direction[1][1] = 0.0;
  wall_length[1] = (MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;


  wall_position[2][0] = -ROAD_WIDTH/2;
  wall_position[2][1] = -ROAD_WIDTH/2-(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_direction[2][0] = 0.0;
  wall_direction[2][1] = 1.0;
  wall_length[2] = (MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;
  wall_position[3][0] = -ROAD_WIDTH/2-(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_position[3][1] = -ROAD_WIDTH/2;
  wall_direction[3][0] = 1.0;
  wall_direction[3][1] = 0.0;
  wall_length[3] = (MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;


  wall_position[4][0] = 0.0;
  wall_position[4][1] = ROAD_WIDTH/2;
  wall_direction[4][0] = 1.0;
  wall_direction[4][1] = 0.0;
  wall_length[4] = MAP_SQUIRE_WIDTH;


  switch( direction ){
  case FROM_EAST:
    /* to rotate 90 deg anti-crockwise*/
    for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
      rotate_90deg_anticlockwise( wall_position[wall_index]);
      rotate_90deg_anticlockwise( wall_direction[wall_index]);
    }
    break;
  case FROM_NORTH:
    /* to rotate 180 deg*/
    for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
      rotate_180deg( wall_position[wall_index]);
      rotate_180deg( wall_direction[wall_index]);
    }
    break;
  case FROM_WEST:
    { /* to rotate 90 deg crockwise*/
      for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
	rotate_90deg_clockwise( wall_position[wall_index]);
	rotate_90deg_clockwise( wall_direction[wall_index]);
      }
    }
    break;
  case FROM_SOUTH:
  default:
    break;
  }

  for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
    make_wall( map[map_x][map_y]->center_position[0] + wall_position[wall_index][0], /* wall center position x */
	       map[map_x][map_y]->center_position[1] + wall_position[wall_index][1], /* wall center position y*/
	       wall_direction[wall_index][0], /* wall direction x */
	       wall_direction[wall_index][1], /* wall direction y */
	       wall_length[wall_index],
	       &(map[map_x][map_y]->wall_object[wall_index]) );
  }


  return 0;
}


int make_cross_road_walls( int map_x, int map_y ){
  static const int wall_size = 8;
  dReal wall_position[wall_size][2];
  dReal wall_direction[wall_size][2];
  dReal wall_length[wall_size];
  int wall_index;
  if( map[map_x][map_y] == NULL ){
    map[map_x][map_y] = (struct MapData*)malloc( sizeof(struct MapData) );
    for( wall_index = 0; wall_index < MAP_ONE_SQUIRE_MAX_WALL_SIZE; ++wall_index ){
      map[map_x][map_y]->wall_object[wall_index].object_type = WALL;
      map[map_x][map_y]->wall_object[wall_index].data = &map[map_x][map_y]->wall_data[wall_index];
    }
  }else{
    return -1; // if the map on the position is filled, then return -1 (error).
  }
  
  map[map_x][map_y]->center_position[0] = ( (dReal)map_x + 0.5 ) * MAP_SQUIRE_WIDTH;
  map[map_x][map_y]->center_position[1] = ( (dReal)map_y + 0.5 ) * MAP_SQUIRE_WIDTH;
  
  map[map_x][map_y]->wall_size = wall_size;
  
  wall_position[0][0] = +ROAD_WIDTH/2;
  wall_position[0][1] = -ROAD_WIDTH/2-(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_direction[0][0] = 0.0;
  wall_direction[0][1] = 1.0;
  wall_length[0] = (MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;

  wall_position[1][0] = +ROAD_WIDTH/2+(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_position[1][1] = -ROAD_WIDTH/2;
  wall_direction[1][0] = 1.0;
  wall_direction[1][1] = 0.0;
  wall_length[1] = (MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;


  wall_position[2][0] = +ROAD_WIDTH/2;
  wall_position[2][1] = +ROAD_WIDTH/2+(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_direction[2][0] = 0.0;
  wall_direction[2][1] = 1.0;
  wall_length[2] = (MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;
  wall_position[3][0] = +ROAD_WIDTH/2+(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_position[3][1] = +ROAD_WIDTH/2;
  wall_direction[3][0] = 1.0;
  wall_direction[3][1] = 0.0;
  wall_length[3] = (MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;


  wall_position[4][0] = -ROAD_WIDTH/2;
  wall_position[4][1] = +ROAD_WIDTH/2+(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_direction[4][0] = 0.0;
  wall_direction[4][1] = 1.0;
  wall_length[4] = (MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;
  wall_position[5][0] = -ROAD_WIDTH/2-(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_position[5][1] = +ROAD_WIDTH/2;
  wall_direction[5][0] = 1.0;
  wall_direction[5][1] = 0.0;
  wall_length[5] = (MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;


  wall_position[6][0] = -ROAD_WIDTH/2;
  wall_position[6][1] = -ROAD_WIDTH/2-(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_direction[6][0] = 0.0;
  wall_direction[6][1] = 1.0;
  wall_length[6] = (MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;
  wall_position[7][0] = -ROAD_WIDTH/2-(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_position[7][1] = -ROAD_WIDTH/2;
  wall_direction[7][0] = 1.0;
  wall_direction[7][1] = 0.0;
  wall_length[7] = (MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;

  for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
    make_wall( map[map_x][map_y]->center_position[0] + wall_position[wall_index][0], /* wall center position x */
	       map[map_x][map_y]->center_position[1] + wall_position[wall_index][1], /* wall center position y*/
	       wall_direction[wall_index][0], /* wall direction x */
	       wall_direction[wall_index][1], /* wall direction y */
	       wall_length[wall_index],
	       &(map[map_x][map_y]->wall_object[wall_index]) );
  }

  return 0;
}

int make_dead_end_walls( int map_x, int map_y, enum DIRECTION direction ){
  static const int wall_size = 3;
  dReal wall_position[wall_size][2];
  dReal wall_direction[wall_size][2];
  dReal wall_length[wall_size];
  int wall_index;
  if( map[map_x][map_y] == NULL ){
    map[map_x][map_y] = (struct MapData*)malloc( sizeof(struct MapData) );
    for( wall_index = 0; wall_index < MAP_ONE_SQUIRE_MAX_WALL_SIZE; ++wall_index ){
      map[map_x][map_y]->wall_object[wall_index].object_type = WALL;
      map[map_x][map_y]->wall_object[wall_index].data = &map[map_x][map_y]->wall_data[wall_index];
    }
  }else{
    return -1; // if the map on the position is filled, then return -1 (error).
  }
  
  map[map_x][map_y]->center_position[0] = ( (dReal)map_x + 0.5 ) * MAP_SQUIRE_WIDTH;
  map[map_x][map_y]->center_position[1] = ( (dReal)map_y + 0.5 ) * MAP_SQUIRE_WIDTH;

  map[map_x][map_y]->wall_size = wall_size;

  /* the original is right turn (from south to east)*/
  wall_position[0][0] = -ROAD_WIDTH/2;
  wall_position[0][1] = -(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_direction[0][0] = 0.0;
  wall_direction[0][1] = 1.0;
  wall_length[0] = ROAD_WIDTH+(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;

  wall_position[1][0] = ROAD_WIDTH/2;
  wall_position[1][1] = -(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/4;
  wall_direction[1][0] = 0.0;
  wall_direction[1][1] = 1.0;
  wall_length[1] = ROAD_WIDTH+(MAP_SQUIRE_WIDTH-ROAD_WIDTH)/2;



  wall_position[2][0] = 0.0;
  wall_position[2][1] = +ROAD_WIDTH/2;
  wall_direction[2][0] = 1.0;
  wall_direction[2][1] = 0.0;
  wall_length[2] = ROAD_WIDTH;

  switch( direction ){
  case FROM_EAST:
    for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
      rotate_90deg_anticlockwise( wall_position[wall_index]);
      rotate_90deg_anticlockwise( wall_direction[wall_index]);
    }
    break;
  case FROM_NORTH:
    for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
      rotate_180deg( wall_position[wall_index]);
      rotate_180deg( wall_direction[wall_index]);
    }
    break;
  case FROM_WEST:
    /* to rotate 90 deg crockwise*/
    for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
      rotate_90deg_clockwise( wall_position[wall_index]);
      rotate_90deg_clockwise( wall_direction[wall_index]);
    }
    break;
  case FROM_SOUTH_TO_EAST:
  case FROM_EAST_TO_SOUTH:
  default:
    break;
  }


  for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
    make_wall( map[map_x][map_y]->center_position[0] + wall_position[wall_index][0], /* wall center position x */
	       map[map_x][map_y]->center_position[1] + wall_position[wall_index][1], /* wall center position y*/
	       wall_direction[wall_index][0], /* wall direction x */
	       wall_direction[wall_index][1], /* wall direction y */
	       wall_length[wall_index],
	       &(map[map_x][map_y]->wall_object[wall_index]) );
  }




  
  return 0;
}

int make_wall( dReal wall_center_position_x, dReal wall_center_position_y,
	       dReal wall_direction_x, dReal wall_direction_y,
	       dReal wall_length,
	       struct GameObject* wall_object ){
  
  dMass box_mass;
  dQuaternion quaternion; // quaternion represents body's rotation.
  dReal theta;
  
  struct WallData* wall_data;
  wall_data = (struct WallData*)(wall_object->data);
  wall_data->body_ID = dBodyCreate( get_ode_world_id() );
  dBodySetPosition ( wall_data->body_ID,
		     wall_center_position_x, wall_center_position_y,
		     WALL_HEIGHT/2.0);
  dMassSetBox(&box_mass,1, wall_length, WALL_WIDTH, WALL_HEIGHT);
  dMassAdjust(&box_mass,WALL_MASS );
  dBodySetMass( wall_data->body_ID, &box_mass );
  wall_data->geom_ID = dCreateBox(get_ode_space_id(), wall_length,WALL_WIDTH,WALL_HEIGHT);
  
  theta = atan2(wall_direction_y, wall_direction_x);
  dQFromAxisAndAngle(quaternion, 0, 0, 1, theta );
  dBodySetQuaternion( wall_data->body_ID, quaternion);
  
  dGeomSetBody( wall_data->geom_ID, wall_data->body_ID );

  wall_data->fixed_joint_ID = dJointCreateFixed( get_ode_world_id(), 0);
  dJointAttach (wall_data->fixed_joint_ID, wall_data->body_ID, 0 );
  dJointSetFixed( wall_data->fixed_joint_ID );

  dGeomSetData( wall_data->geom_ID,
		wall_object );
  return 0;
}

void visualize_map(){
  int map_x, map_y;
  int wall_index;

  dsSetTexture (DS_WOOD);
  dsSetColor(1.0,0.0,0.0);
 
  for(map_x = 0; map_x < MAP_X_SIZE; ++map_x){
    for(map_y = 0; map_y < MAP_Y_SIZE; ++map_y){
      if( map[map_x][map_y] != NULL ){
        for( wall_index = 0; wall_index < map[map_x][map_y]->wall_size; ++wall_index ){
          dVector3 lengths;
          dGeomBoxGetLengths(map[map_x][map_y]->wall_data[wall_index].geom_ID, lengths);
          dsDrawBox( dGeomGetPosition(map[map_x][map_y]->wall_data[wall_index].geom_ID),
		     dGeomGetRotation(map[map_x][map_y]->wall_data[wall_index].geom_ID),
		     lengths);
        }
      }
    }
  }  
}

void rotate_90deg_clockwise( dReal* vec2 ){
  dReal x,y;
  x = vec2[0];
  y = vec2[1];
  vec2[0] = y;
  vec2[1] = -x;
}
void rotate_90deg_anticlockwise(dReal* vec2 ){
  dReal x,y;
  x = vec2[0];
  y = vec2[1];
  vec2[0] = -y;
  vec2[1] = x;
}
void rotate_180deg( dReal* vec2 ){
  dReal x,y;
  x = vec2[0];
  y = vec2[1];
  vec2[0] = -x;
  vec2[1] = -y;
}
