#include <gamelib/gamelib.h>
#include <gamelib/map.h>
#include <gamelib/self_agent.h>
#include <stdbool.h>
#include <assert.h>

int self_agent_initialized = false; /* false: not initialized. true: initialized */
static struct GameObject self_agent;
static struct SelfAgentData self_agent_data;

static struct GameObject self_agent_bullets[SELF_AGENT_BULLET_NUMBER];
static struct SelfAgentBulletData self_agent_bullets_data[SELF_AGENT_BULLET_NUMBER];

//enum ObjectType self_agent_type = SELF_AGENT;
//enum ObjectType self_agent_bullet_type = SELF_AGENT_BULLET;

void initialize_self_agent( int map_x, int map_y,
		     int initial_hit_point,
		     dReal initial_direction_angle ){ /* EAST direction is 0 [rad]*/
  int bullet_index;
  dReal* self_agent_position;
  dMass mass;
  assert( false == self_agent_initialized );
  
  self_agent_data.body_ID = dBodyCreate( get_ode_world_id() );
  dBodySetPosition( self_agent_data.body_ID,
  		    ( (dReal)map_x + 0.5 ) * MAP_SQUIRE_WIDTH, /* x position */
  		    ( (dReal)map_y + 0.5 ) * MAP_SQUIRE_WIDTH, /* y position */
  		    SELF_AGENT_SPHERE_RADIUS ); /* z position */
  
  dMassSetSphere( &mass, 1, SELF_AGENT_SPHERE_RADIUS );
  dMassAdjust( &mass, SELF_AGENT_MASS );
  dBodySetMass( self_agent_data.body_ID, &mass );

  self_agent_data.geom_ID = dCreateSphere( get_ode_space_id(), SELF_AGENT_SPHERE_RADIUS );
  dGeomSetBody( self_agent_data.geom_ID, self_agent_data.body_ID);
  self_agent.object_type = SELF_AGENT;
  self_agent.data = &self_agent_data;
  dGeomSetData( self_agent_data.geom_ID, &self_agent );

  self_agent_data.hit_point = initial_hit_point;
  self_agent_data.direction_angle = initial_direction_angle; 
  self_agent_data.pulling_force[0] = 0.0;
  self_agent_data.pulling_force[1] = 0.0;
  self_agent_data.pulling_force[2] = 0.0;
  
  for( bullet_index = 0; bullet_index < SELF_AGENT_BULLET_NUMBER; ++bullet_index ){
    self_agent_bullets_data[ bullet_index ].body_ID = dBodyCreate( get_ode_world_id() );
    dBodySetPosition( self_agent_bullets_data[ bullet_index ].body_ID,
		      0.0, 0.0, 0.0 ); /* set position (x,y,z)=(0.0,0.0,0.0)*/
    dMassSetSphere( &mass, 1, SELF_AGENT_BULLET_SPHERE_RADIUS );
    dMassAdjust( &mass, SELF_AGENT_BULLET_MASS );
    dBodySetMass( self_agent_bullets_data[bullet_index].body_ID, &mass );
    
    self_agent_bullets_data[ bullet_index ].geom_ID =
      dCreateSphere( get_ode_space_id(),SELF_AGENT_BULLET_SPHERE_RADIUS );
    
    dGeomSetBody( self_agent_bullets_data[bullet_index].geom_ID,
		  self_agent_bullets_data[bullet_index].body_ID);

    self_agent_bullets[bullet_index].object_type = SELF_AGENT_BULLET;
    self_agent_bullets[bullet_index].data = &self_agent_bullets_data[bullet_index];
    dGeomSetData( self_agent_bullets_data[bullet_index].geom_ID,
		  &(self_agent_bullets[bullet_index]) );

    dBodyDisable( self_agent_bullets_data[bullet_index].body_ID );
    self_agent_bullets_data[bullet_index].countdown_timer = 0;
  }

  self_agent_initialized = true;
}


void visualize_self_agent(){
  assert( true == self_agent_initialized );
  //dsSetTexture (DS_WOOD);
  dsSetTexture(DS_CHECKERED);
  dsSetColor(0.0,0.0,1.0);

  dsSetSphereQuality(2);
  dsDrawSphere (dGeomGetPosition( self_agent_data.geom_ID ),
		dGeomGetRotation( self_agent_data.geom_ID ),
		SELF_AGENT_SPHERE_RADIUS );
}

void update_self_agent(){
  const dReal *agent_velocity;
  const dReal *agent_angular_velocity;
  int bullet_index;
  assert( true == self_agent_initialized );
  
  dBodyAddForce(self_agent_data.body_ID,
		self_agent_data.pulling_force[0],
		self_agent_data.pulling_force[1],
		self_agent_data.pulling_force[2]);
  
  agent_velocity=dBodyGetLinearVel( self_agent_data.body_ID );
  agent_angular_velocity=dBodyGetAngularVel( self_agent_data.body_ID );

  /* dumping of the movement */
  self_agent_data.pulling_force[0] = -agent_velocity[0];
  self_agent_data.pulling_force[1] = -agent_velocity[1];
  self_agent_data.pulling_force[2] = -agent_velocity[2];

  for( bullet_index = 0; bullet_index < SELF_AGENT_BULLET_NUMBER; bullet_index++ ){
    if( dBodyIsEnabled ( self_agent_bullets_data[ bullet_index ].body_ID ) ){
      if( 0 < self_agent_bullets_data[ bullet_index ].countdown_timer ){
	self_agent_bullets_data[ bullet_index ].countdown_timer--;
      }else{
	dBodyDisable( self_agent_bullets_data[ bullet_index ].body_ID );
      }
    }
  }
}

void set_first_person_view_point(){
  const dReal *agent_position;
  
  float camera_position[3];
  float camera_angle;
  float heading_pitch_roll[3];

  assert( true == self_agent_initialized );


  agent_position = dBodyGetPosition( self_agent_data.body_ID );
  camera_position[0] = (float)agent_position[0] - SELF_AGENT_SPHERE_RADIUS * cos( self_agent_data.direction_angle );
  camera_position[1] = (float)agent_position[1] - SELF_AGENT_SPHERE_RADIUS * sin( self_agent_data.direction_angle );
  camera_position[2] = (float)agent_position[2]+2*SELF_AGENT_SPHERE_RADIUS;
  
  heading_pitch_roll[0] = self_agent_data.direction_angle * 180.0/M_PI; /* heading (direction) */
  heading_pitch_roll[1] = 0.0; /* pitch */
  heading_pitch_roll[2] = 0.0; /* roll */
  /*
  printf("camera_angle=%e  ", (double)camera_angle*180.0/M_PI);
  printf("camera_position={%e,%e,%e}\n",
	 camera_position[0],
	 camera_position[1],
	 camera_position[2]);
  printf("heading_pitch_roll={%e,%e,%e}\n",
	 heading_pitch_roll[0], heading_pitch_roll[1], heading_pitch_roll[2]);
  */

  dsSetViewpoint(camera_position,
  		 heading_pitch_roll);
 

}

void set_behind_view_point(){
  const dReal *agent_position;
  
  float camera_position[3];
  float camera_angle;
  float heading_pitch_roll[3];
  assert( true == self_agent_initialized );

  
  dBodyAddForce(self_agent_data.body_ID,
		self_agent_data.pulling_force[0],
		self_agent_data.pulling_force[1],
		self_agent_data.pulling_force[2]);
  
  agent_position = dBodyGetPosition( self_agent_data.body_ID );
  camera_position[0] = (float)agent_position[0] - 6*SELF_AGENT_SPHERE_RADIUS * cos( self_agent_data.direction_angle );
  camera_position[1] = (float)agent_position[1] - 6*SELF_AGENT_SPHERE_RADIUS * sin( self_agent_data.direction_angle );
  camera_position[2] = (float)agent_position[2]+15*SELF_AGENT_SPHERE_RADIUS;
  
  heading_pitch_roll[0] = self_agent_data.direction_angle * 180.0/M_PI; /* heading (direction) */
  heading_pitch_roll[1] = -60.0; /* pitch */
  heading_pitch_roll[2] = 0.0; /* roll */
  /*
  printf("camera_angle=%e  ", (double)camera_angle*180.0/M_PI);
  printf("camera_position={%e,%e,%e}\n",
	 camera_position[0],
	 camera_position[1],
	 camera_position[2]);
  printf("heading_pitch_roll={%e,%e,%e}\n",
	 heading_pitch_roll[0], heading_pitch_roll[1], heading_pitch_roll[2]);
  */

  dsSetViewpoint(camera_position,
  		 heading_pitch_roll);
 

}


void command_self_agent( int command ){
  const dReal* position;
  assert( true == self_agent_initialized );

  switch( command ){
  case 'f': /* Go right side way*/
    {
      self_agent_data.pulling_force[0] += PULLING_FORCE_SIZE * cos(self_agent_data.direction_angle-M_PI/2);
      self_agent_data.pulling_force[1] += PULLING_FORCE_SIZE * sin(self_agent_data.direction_angle-M_PI/2);
      self_agent_data.pulling_force[2] += 0.0;
    }
    break;
  case 'd': /* Go forward*/
    {
      self_agent_data.pulling_force[0] += PULLING_FORCE_SIZE * cos(self_agent_data.direction_angle);
      self_agent_data.pulling_force[1] += PULLING_FORCE_SIZE * sin(self_agent_data.direction_angle);
      self_agent_data.pulling_force[2] += 0.0;
    }
    break;
  case 's': /* Go backward*/
    {
    self_agent_data.pulling_force[0] += PULLING_FORCE_SIZE * cos(self_agent_data.direction_angle+M_PI);
    self_agent_data.pulling_force[1] += PULLING_FORCE_SIZE * sin(self_agent_data.direction_angle+M_PI);
    self_agent_data.pulling_force[2] += 0.0;    
    }
    break;
  case 'a': /* Go left side way*/
    {
      self_agent_data.pulling_force[0] += PULLING_FORCE_SIZE * cos(self_agent_data.direction_angle+M_PI/2);
      self_agent_data.pulling_force[1] += PULLING_FORCE_SIZE * sin(self_agent_data.direction_angle+M_PI/2);
      self_agent_data.pulling_force[2] += 0.0;
    }
    break;
  case 'e': /* Trun right*/
    {
      self_agent_data.direction_angle -= TURNING_STEP;
      if( self_agent_data.direction_angle < -M_PI ){
	self_agent_data.direction_angle += 2*M_PI;
      }
    }
    break;
  case 'w': /* Turn left*/
    {
      self_agent_data.direction_angle += TURNING_STEP;
      if( M_PI < self_agent_data.direction_angle ){
	self_agent_data.direction_angle -= 2*M_PI;
      }
    }
    break;
  case 'g': /* Jump*/
    {
      position = dGeomGetPosition( self_agent_data.geom_ID );
      if( position[2] < SELF_AGENT_SPHERE_RADIUS * 2){
	self_agent_data.pulling_force[0] += 0.0;
	self_agent_data.pulling_force[1] += 0.0;
	self_agent_data.pulling_force[2] += JUMPING_FORCE_SIZE;
      }
    }
    break;
  case 'l': /* Shoot the gun*/
    self_agent_shoot_gun();
  }
}

void self_agent_shoot_gun(){
  assert( true == self_agent_initialized );

  int bullet_index;
  const dReal* position;
  dVector3 initial_velocity;
  for( bullet_index = 0; bullet_index < SELF_AGENT_BULLET_NUMBER; bullet_index++ ){
    if( !dBodyIsEnabled ( self_agent_bullets_data[ bullet_index ].body_ID ) ){
      position = dBodyGetPosition( self_agent_data.body_ID );
      dBodySetPosition( self_agent_bullets_data[ bullet_index ].body_ID,
			position[0], position[1], position[2] );

      initial_velocity[0] = SELF_AGENT_BULLET_INITIAL_VELOCITY * cos( self_agent_data.direction_angle );
      initial_velocity[1] = SELF_AGENT_BULLET_INITIAL_VELOCITY * sin( self_agent_data.direction_angle );
      initial_velocity[2] = SELF_AGENT_BULLET_INITIAL_VELOCITY * sin( SELF_AGENT_BULLET_INITIAL_ANGLE );
      

      dBodySetLinearVel (self_agent_bullets_data[ bullet_index ].body_ID,
			 initial_velocity[0],
			 initial_velocity[1],
			 initial_velocity[2]);
      dBodySetAngularVel( self_agent_bullets_data[ bullet_index ].body_ID,
			  0.0,0.0,0.0);
      dBodyEnable( self_agent_bullets_data[ bullet_index ].body_ID );
      self_agent_bullets_data[ bullet_index ].countdown_timer = (int)((double)SELF_AGENT_BULLET_SURVIVAL_TIME/(double)SIMULATION_TIME_STEP);
      break;
    }
  }
}

void destroy_self_agent(){
  assert( true == self_agent_initialized );

  int bullet_index;
  for( bullet_index = 0; bullet_index < SELF_AGENT_BULLET_NUMBER; bullet_index++ ){
    if( dBodyIsEnabled ( self_agent_bullets_data[ bullet_index ].body_ID ) ){
      dGeomSetData( self_agent_bullets_data[bullet_index].geom_ID, NULL );
      dBodyDestroy( self_agent_bullets_data[bullet_index].body_ID );
      dGeomDestroy( self_agent_bullets_data[bullet_index].geom_ID );
    }
  }


  dGeomSetData( self_agent_data.geom_ID, NULL );
  dBodyDestroy( self_agent_data.body_ID );
  dGeomDestroy( self_agent_data.geom_ID );
  self_agent_initialized = false;
  printf("destroy self agent: %d\n", self_agent_initialized);
}

void visualize_self_agent_bullets(){
  assert( self_agent_initialized == true );

  int bullet_index;
  dsSetColor(0.0,1.0,1.0);
  for( bullet_index = 0; bullet_index < SELF_AGENT_BULLET_NUMBER; bullet_index++ ){
    if( dBodyIsEnabled ( self_agent_bullets_data[ bullet_index ].body_ID ) ){
      dsDrawSphere (dGeomGetPosition( self_agent_bullets_data[bullet_index].geom_ID ),
		    dGeomGetRotation( self_agent_bullets_data[bullet_index].geom_ID ),
		    SELF_AGENT_BULLET_SPHERE_RADIUS );
    }
  }
}

const dReal* get_self_agent_position(){
  assert( self_agent_initialized == true );
  
  return dBodyGetPosition( self_agent_data.body_ID );
}

int get_self_agent_hit_point(){
  assert( self_agent_initialized == true );
  return self_agent_data.hit_point;
}

void change_self_agent_hit_point( int hit_point ){
  assert( self_agent_initialized == true );
  self_agent_data.hit_point += hit_point;
}

